package clustersim;

import java.io.File;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Scanner;

import logging.ParallelLog;
import logging.StoreLog;
import logging.StoreLogAmazonS3;
import logging.StoreLogConsole;
import logging.StoreLogFile;
import rsalgos.AKG_RSalgorithm;
import rsalgos.KG_RSalgorithm;

import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3Client;
/**
 * The class to run simulations. Can use console, files or Amazon S3 for logging.
 * @author Bogumil Kaminski & Przemyslaw Szufel
 */
public class _Run_Sims {
	public static final int APP_VERSION = 8;
	/**
	 * Runs simulations. The code checks the number of CPU cores available to the JVM and adjust the number of threads accordingly
	 * @param args <ul>
	 * <li> seedsMaster - the master random seed
	 * <li> seedsSpace - the distance between random number seeds for different threads
	 * <li> repetitions - number of simulation repetitions 
	 * <li> target - place to write logs
	 *     <ul>
	 *        <li>"console" - logs will be written to the console
	 *        <li>path starting with drive name or . or /  - logs will be written to the given folder
	 *        <li>S3 bucket name (any other text) - logs will be written to Amazon S3 bucket with the given name. Using S3 requires seting S3 credentials in the source code. 
	 *     </ul> 
	 * </ul>
	 * @throws Exception
	 */
	public static void main(String[] args) throws Exception {
		long seedsMaster = Long.parseLong(args[0]);
		long seedsSpace = Long.parseLong(args[1]);
		long repetitions = Long.parseLong(args[2]);
		String target = args[3];
		StoreLog storeLog = null;
		String hostname = InetAddress.getLocalHost().getHostName();
		
		String versionStr = "v"+String.format("%05d",APP_VERSION);
		
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd-HH-mm-ss");
		
		String fill = "------------------";
		String hos = hostname.substring(0,hostname.contains(".")?hostname.indexOf('.'):hostname.length());
		if (hos.length() < fill.length()) {
			fill = fill.substring(0,fill.length()-hos.length());
		} else {
			fill = "";
		}
		
		String seed_date_hostname = "s"+String.format("%09d", seedsMaster)+"@"+sdf.format(new Date())+"@"+fill+hostname;
				
		if (target.equals("console") ) {
			storeLog = new StoreLogConsole();
		} else if (target.startsWith(".") || target.startsWith("/") || target.charAt(1)==':') {
			storeLog = new StoreLogFile(new File(target));
		} else {			
			AmazonS3 s3Client = new AmazonS3Client(new BasicAWSCredentials("*******************","*********************"));			
			URL url = new URL("http://169.254.169.254/latest/meta-data/hostname");
			InputStream is = url.openStream();
			Scanner s = new Scanner(is);
		    if (s.hasNext()) {
		    	hostname = s.next();
		    }
		    s.close();		    
			storeLog = new StoreLogAmazonS3(s3Client, target, versionStr+"/"+ seed_date_hostname, "us-west-2");
		}		
		ParallelLog log = new ParallelLog("akg@"+versionStr+"@"+seed_date_hostname+"@", storeLog, 1000000, 1, 1,hostname);
		SimThread ts[] = new SimThread[Runtime.getRuntime().availableProcessors()-3];
		for (int i=0;i<ts.length;i++) {
			ts[i] = new SimThread("T"+i,seedsMaster+seedsSpace*i, repetitions,log);
			System.out.println("T"+i+" seed="+(seedsMaster+seedsSpace*i)+" reps="+repetitions);
			ts[i].start();
		}		
		for (int i=0;i<ts.length;i++) {
			ts[i].join();
		}
		log.close();
	}
	
	
	static class SimThread extends Thread {
		private long startSeed;
		private long repetitions;
		private ParallelLog log;
		private String name;
		public SimThread (String name, long startSeed, long repetitions,ParallelLog log) {
			super(name);
			this.name = name;
			this.startSeed = startSeed;
			this.repetitions = repetitions;
			this.log = log;
		}
		public SimThread (long startSeed, long repetitions,String name,ParallelLog log) {
			this.startSeed = startSeed;
			this.repetitions = repetitions;
			this.name = name;
			this.log = log;
		}
		public void run() {
			for (long seed = startSeed;seed<startSeed+repetitions;seed++) {
				long start = System.currentTimeMillis();
				for (int N : new int[]{5,10,20}) {
					int numWorkers = 1;
					CompClusterEmulator model1 = new CompClusterEmulator(seed, N, numWorkers, new KG_RSalgorithm(),false);
					CompClusterEmulator.run (model1, 95,false);								
					log.log(seed+"\t"+"N="+"\t"+N+"\t"+"KG"+"\t"+numWorkers+"\t"+model1.getPointSelectionHistory()+"\t"+model1.getOptimalSolutionDistanceBinHistory()+"\t"+model1.getOptimalSolutionDistanceDoubleHistory());
					for (numWorkers=5;numWorkers<=5;numWorkers++) {
						CompClusterEmulator model = new CompClusterEmulator(seed, N, numWorkers, new AKG_RSalgorithm(),false);
						CompClusterEmulator.run (model, 95,false);		
						log.log(seed+"\t"+"N="+"\t"+N+"\t"+"AKG"+"\t"+numWorkers+"\t"+model.getPointSelectionHistory()+"\t"+model.getOptimalSolutionDistanceBinHistory()+"\t"+model.getOptimalSolutionDistanceDoubleHistory());
					}
				}
				System.out.println(this.name+" Seed "+seed+ " took "+((System.currentTimeMillis()-start)/1000.0)+" seconds.");
			}
			
        }
		
		
	}

}
